from flask import Flask, jsonify, render_template_string, request
from flask_limiter import Limiter
from flask_limiter.util import get_remote_address
import os
import time
from ringcentral import SDK
import json
from datetime import datetime, timezone
from functools import wraps
import logging

# Configure logging for better debugging
logging.basicConfig(level=logging.DEBUG, format='%(asctime)s - %(levelname)s - %(message)s')
logger = logging.getLogger(__name__)

app = Flask(__name__)

# Security configurations
app.config['SECRET_KEY'] = os.environ.get('SECRET_KEY', 'change-this-secret-key-in-production')

# Rate limiting to prevent abuse
limiter = Limiter(
    key_func=get_remote_address,
    default_limits=["100 per hour", "10 per minute"]
)
limiter.init_app(app)

# RingCentral configuration - move to environment variables in production
config = {
    "client_id": "1CayFASqF2Qdv1MFI2nDgo",
    "client_secret": "Xim1wYJpsORcp3d4KrwrjHZr9t2ckc8H9cvPkhtmHxpV",
    "server_url": "https://platform.ringcentral.com",
    "jwt": {
        "Reports": "eyJraWQiOiI4NzYyZjU5OGQwNTk0NGRiODZiZjVjYTk3ODA0NzYwOCIsInR5cCI6IkpXVCIsImFsZyI6IlJTMjU2In0.eyJhdWQiOiJodHRwczovL3BsYXRmb3JtLnJpbmdjZW50cmFsLmNvbS9yZXN0YXBpL29hdXRoL3Rva2VuIiwic3ViIjoiMjIzNzIxMDQyIiwiaXNzIjoiaHR0cHM6Ly9wbGF0Zm9ybS5yaW5nY2VudHJhbC5jb20iLCJleHAiOjIyNDAyNjU1OTksImlhdCI6MTcwMTEwMzI5MywianRpIjoiYkxGZDJqOTVSUWk2THROcW1JLVhFZyJ9.ALtTujcoO5CNQmkh-YX-tMibPE2Szx3Kzt8PsEKPetygnW23HCpKtaXufDjgG_qMe5PxfcuGQgLITU3gGh_H4k4kSIlviyJTrWwiwwOcBTsorNbIQMjbVwNdpMZGfzCf1bqkAFq7FIJMfR0KopXSBSL5RsnkWa12Vm_uUEQBIpiZd0T0F9uVJeaVcQlOPr4dK1zae3XEtqubdWT6xyAqli-glTj4IDgn-VNfM2WGk63QMb6ApraALmwulNCjqNPK6bJ7Q2U65zMPRDJmfy-jVpdS7KABBuj76bt0itBwt_-EdYp92DjB8-aHEsJiaLOYMW2bKuAqq95hHlJbnQ45fQ"
    }
}

hidden_users = ["Emma Crane", "Nathan Oliver", "Fran Ferreria"]

def format_duration(seconds):
    """Convert seconds to HH:MM:SS format"""
    if seconds < 0:
        seconds = 0
    
    hours = seconds // 3600
    minutes = (seconds % 3600) // 60
    secs = seconds % 60
    
    return f"{hours:02d}:{minutes:02d}:{secs:02d}"

def get_best_duration(record):
    """Extract the best duration from a call record"""
    # Try different duration fields in order of preference
    duration_fields = [
        ('duration', record.get('duration', 0)),
        ('billableDuration', record.get('billableDuration', 0)),
        ('talkTime', record.get('talkTime', 0))
    ]
    
    # Calculate legs duration
    legs_duration = 0
    legs = record.get('legs', [])
    if legs:
        for leg in legs:
            leg_duration = leg.get('duration', 0)
            if isinstance(leg_duration, (int, float)) and leg_duration > 0:
                legs_duration += leg_duration
    
    duration_fields.append(('legs_total', legs_duration))
    
    # Log all duration options for debugging
    user_name = record.get('from', {}).get('name', 'Unknown')
    logger.debug(f"Duration options for {user_name}:")
    for field_name, value in duration_fields:
        logger.debug(f"  {field_name}: {value}s ({format_duration(value)})")
    
    # Choose the best duration (prefer record duration if > 0, otherwise use legs)
    best_duration = 0
    for field_name, value in duration_fields:
        if isinstance(value, (int, float)) and value > 0:
            best_duration = max(best_duration, value)
    
    logger.debug(f"  Selected duration: {best_duration}s ({format_duration(best_duration)})")
    return int(best_duration)

def fetch_call_logs():
    """Fetch call logs from RingCentral API"""
    try:
        logger.info("Starting to fetch call logs...")
        
        rcsdk = SDK(config["client_id"], config["client_secret"], config["server_url"])
        platform = rcsdk.platform()
        
        # Login with JWT
        logger.info("Authenticating with RingCentral...")
        platform.login(jwt=config["jwt"]["Reports"])
        
        # Get today's date in ISO format
        today_iso = datetime.utcnow().strftime('%Y-%m-%dT00:00:00.000Z')
        logger.info(f"Fetching calls from: {today_iso}")
        
        # Retry logic for rate limiting
        max_retries = 5
        retry_delay = 1
        response = None
        
        for attempt in range(max_retries):
            try:
                logger.info(f"API call attempt {attempt + 1}/{max_retries}")
                response = platform.get('/restapi/v1.0/account/~/call-log', {
                    'dateFrom': today_iso,
                    'type': 'Voice',
                    'direction': 'Outbound',
                    'perPage': 1000  # Increase page size to get more records
                })
                logger.info("API call successful")
                break
            except Exception as e:
                logger.error(f"API call failed: {str(e)}")
                if 'Request rate exceeded' in str(e):
                    logger.info(f"Rate limited, waiting {retry_delay} seconds...")
                    time.sleep(retry_delay)
                    retry_delay *= 2
                else:
                    raise e
        
        if response is None:
            raise Exception("Max retries exceeded")
            
        call_logs = json.loads(response.text())
        
        # Log API response info
        total_records = len(call_logs.get('records', []))
        logger.info(f"Retrieved {total_records} total call records")
        
        # Debug: Log structure of first record
        if call_logs.get('records'):
            sample = call_logs['records'][0]
            logger.debug("Sample record structure:")
            logger.debug(json.dumps(sample, indent=2, default=str))
        
        user_data = {}
        processed_records = 0
        
        for record in call_logs.get('records', []):
            # Only process outbound calls
            if record.get('direction') != 'Outbound':
                continue
                
            # Extract user information
            from_info = record.get('from', {})
            user_name = from_info.get('name')
            
            # Skip if no name or in hidden users
            if not user_name or user_name in hidden_users:
                continue
            
            # Get call duration
            duration = get_best_duration(record)
            
            # Get start time
            start_time_str = record.get('startTime')
            if not start_time_str:
                logger.warning(f"No start time for record: {record.get('id', 'Unknown')}")
                continue
                
            try:
                start_time = datetime.fromisoformat(start_time_str.replace("Z", "+00:00"))
            except ValueError as e:
                logger.error(f"Invalid start time format: {start_time_str}, error: {e}")
                continue
            
            # Initialize user data if not exists
            if user_name not in user_data:
                user_data[user_name] = {
                    "total_duration": 0,
                    "last_call_time": start_time,
                    "call_count": 0
                }
            
            # Update user data
            user_data[user_name]["total_duration"] += duration
            user_data[user_name]["call_count"] += 1
            
            # Update last call time if this call is more recent
            if start_time > user_data[user_name]["last_call_time"]:
                user_data[user_name]["last_call_time"] = start_time
            
            processed_records += 1
        
        logger.info(f"Processed {processed_records} outbound call records")
        logger.info(f"Found {len(user_data)} unique users")
        
        # Format results
        results = []
        now = datetime.now(timezone.utc)
        
        # Log user totals for debugging
        logger.info("User call totals:")
        for user, data in sorted(user_data.items(), key=lambda x: -x[1]["total_duration"]):
            total_duration = data["total_duration"]
            call_count = data["call_count"]
            
            logger.info(f"  {user}: {call_count} calls, {total_duration}s total ({format_duration(total_duration)})")
            
            # Format total time
            total_time_str = format_duration(total_duration)
            
            # Calculate time since last call
            time_diff = now - data["last_call_time"]
            minutes_ago = int(time_diff.total_seconds() // 60)
            
            if minutes_ago < 1:
                last_call_str = "just now"
            elif minutes_ago < 60:
                last_call_str = f"{minutes_ago} minute{'s' if minutes_ago != 1 else ''} ago"
            else:
                hours_ago = minutes_ago // 60
                remaining_minutes = minutes_ago % 60
                if remaining_minutes == 0:
                    last_call_str = f"{hours_ago} hour{'s' if hours_ago != 1 else ''} ago"
                else:
                    last_call_str = f"{hours_ago}h {remaining_minutes}m ago"
            
            results.append({
                "name": user,
                "total_time": total_time_str,
                "call_count": call_count,
                "last_call_ago": last_call_str
            })
        
        logger.info(f"Returning {len(results)} user records")
        return results
        
    except Exception as e:
        logger.error(f"Error fetching call logs: {str(e)}")
        raise Exception(f"Failed to fetch call logs: {str(e)}")

@app.route('/')
def dashboard():
    """Serve the dashboard HTML"""
    return render_template_string(r"""
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Call Logs Dashboard</title>
        <style>
            * {
                margin: 0;
                padding: 0;
                box-sizing: border-box;
            }

            body {
                font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                min-height: 100vh;
                padding: 20px;
            }

            .container {
                max-width: 1200px;
                margin: 0 auto;
            }

            .header {
                text-align: center;
                margin-bottom: 40px;
                color: white;
            }

            .header h1 {
                font-size: 2.5rem;
                font-weight: 300;
                margin-bottom: 10px;
                text-shadow: 0 2px 4px rgba(0,0,0,0.3);
            }

            .header p {
                font-size: 1.1rem;
                opacity: 0.9;
            }

            .auto-refresh-info {
                font-size: 0.9rem;
                opacity: 0.8;
                margin-top: 5px;
            }

            .stats-grid {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
                gap: 20px;
                margin-bottom: 40px;
            }

            .stat-card {
                background: rgba(255, 255, 255, 0.95);
                backdrop-filter: blur(10px);
                border-radius: 15px;
                padding: 25px;
                text-align: center;
                box-shadow: 0 8px 32px rgba(0,0,0,0.1);
                border: 1px solid rgba(255, 255, 255, 0.2);
                transition: transform 0.3s ease, box-shadow 0.3s ease;
            }

            .stat-card:hover {
                transform: translateY(-5px);
                box-shadow: 0 12px 40px rgba(0,0,0,0.15);
            }

            .stat-value {
                font-size: 2rem;
                font-weight: bold;
                color: #667eea;
                margin-bottom: 5px;
            }

            .stat-label {
                color: #666;
                font-size: 0.9rem;
                text-transform: uppercase;
                letter-spacing: 1px;
            }

            .call-logs-container {
                background: rgba(255, 255, 255, 0.95);
                backdrop-filter: blur(10px);
                border-radius: 20px;
                padding: 30px;
                box-shadow: 0 20px 60px rgba(0,0,0,0.1);
                border: 1px solid rgba(255, 255, 255, 0.2);
            }

            .controls {
                display: flex;
                justify-content: space-between;
                align-items: center;
                margin-bottom: 25px;
                flex-wrap: wrap;
                gap: 15px;
            }

            .section-title {
                font-size: 1.5rem;
                color: #333;
                display: flex;
                align-items: center;
                gap: 10px;
                margin: 0;
            }

            .section-title::before {
                content: '📞';
                font-size: 1.2rem;
            }

            .refresh-controls {
                display: flex;
                align-items: center;
                gap: 15px;
            }

            .refresh-btn {
                background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
                color: white;
                border: none;
                padding: 12px 24px;
                border-radius: 25px;
                cursor: pointer;
                font-size: 0.9rem;
                font-weight: 500;
                transition: all 0.3s ease;
                box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
            }

            .refresh-btn:hover {
                transform: translateY(-2px);
                box-shadow: 0 6px 20px rgba(102, 126, 234, 0.4);
            }

            .refresh-btn:disabled {
                opacity: 0.6;
                cursor: not-allowed;
                transform: none;
            }

            .auto-refresh-toggle {
                display: flex;
                align-items: center;
                gap: 8px;
                font-size: 0.9rem;
                color: #666;
            }

            .toggle-switch {
                position: relative;
                width: 44px;
                height: 24px;
                background: #ccc;
                border-radius: 12px;
                cursor: pointer;
                transition: background 0.3s;
            }

            .toggle-switch.active {
                background: #667eea;
            }

            .toggle-slider {
                position: absolute;
                top: 2px;
                left: 2px;
                width: 20px;
                height: 20px;
                background: white;
                border-radius: 50%;
                transition: transform 0.3s;
            }

            .toggle-switch.active .toggle-slider {
                transform: translateX(20px);
            }

            .call-log-item {
                display: flex;
                justify-content: space-between;
                align-items: center;
                padding: 20px;
                margin-bottom: 15px;
                background: linear-gradient(135deg, #f8f9ff 0%, #f0f4ff 100%);
                border-radius: 12px;
                border-left: 4px solid #667eea;
                transition: all 0.3s ease;
                position: relative;
                overflow: hidden;
            }

            .call-log-item:hover {
                transform: translateX(5px);
                box-shadow: 0 5px 20px rgba(102, 126, 234, 0.15);
            }

            .call-log-item::before {
                content: '';
                position: absolute;
                top: 0;
                left: 0;
                right: 0;
                bottom: 0;
                background: linear-gradient(135deg, rgba(102, 126, 234, 0.05) 0%, rgba(118, 75, 162, 0.05) 100%);
                opacity: 0;
                transition: opacity 0.3s ease;
            }

            .call-log-item:hover::before {
                opacity: 1;
            }

            .user-info {
                flex: 1;
                z-index: 1;
                position: relative;
            }

            .user-name {
                font-size: 1.1rem;
                font-weight: 600;
                color: #333;
                margin-bottom: 5px;
            }

            .call-stats {
                display: flex;
                gap: 20px;
                align-items: center;
                z-index: 1;
                position: relative;
            }

            .stat-item {
                display: flex;
                flex-direction: column;
                align-items: center;
                text-align: center;
                min-width: 80px;
            }

            .stat-item-value {
                font-weight: 600;
                color: #667eea;
                font-size: 1rem;
            }

            .stat-item-label {
                font-size: 0.8rem;
                color: #666;
                margin-top: 2px;
            }

            .loading {
                display: flex;
                justify-content: center;
                align-items: center;
                padding: 40px;
                color: #666;
            }

            .loading::before {
                content: '';
                width: 20px;
                height: 20px;
                border: 2px solid #f3f3f3;
                border-top: 2px solid #667eea;
                border-radius: 50%;
                animation: spin 1s linear infinite;
                margin-right: 10px;
            }

            @keyframes spin {
                0% { transform: rotate(0deg); }
                100% { transform: rotate(360deg); }
            }

            .no-data {
                text-align: center;
                padding: 40px;
                color: #666;
                font-style: italic;
            }

            .error {
                text-align: center;
                padding: 40px;
                color: #e74c3c;
                background: rgba(231, 76, 60, 0.1);
                border-radius: 8px;
                margin: 20px 0;
            }

            .last-updated {
                text-align: center;
                font-size: 0.8rem;
                color: #666;
                margin-top: 20px;
            }

            @media (max-width: 768px) {
                .call-log-item {
                    flex-direction: column;
                    gap: 15px;
                    text-align: center;
                }

                .call-stats {
                    justify-content: center;
                }

                .header h1 {
                    font-size: 2rem;
                }

                .controls {
                    flex-direction: column;
                    align-items: stretch;
                }

                .refresh-controls {
                    justify-content: center;
                }
            }
        </style>
    </head>
    <body>
        <div class="container">
            <div class="header">
                <h1>Call Logs Dashboard</h1>
                <p>Today's Outbound Call Activity</p>
                <div class="auto-refresh-info">Auto-refreshes every 2 minutes</div>
            </div>

            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-value" id="totalCalls">0</div>
                    <div class="stat-label">Total Calls</div>
                </div>
                <div class="stat-card">
                    <div class="stat-value" id="totalTime">00:00:00</div>
                    <div class="stat-label">Total Talk Time</div>
                </div>
                <div class="stat-card">
                    <div class="stat-value" id="activeUsers">0</div>
                    <div class="stat-label">Active Users</div>
                </div>
                <div class="stat-card">
                    <div class="stat-value" id="avgCallTime">00:00:00</div>
                    <div class="stat-label">Avg Call Time</div>
                </div>
            </div>

            <div class="call-logs-container">
                <div class="controls">
                    <h2 class="section-title">Call Activity by User</h2>
                    <div class="refresh-controls">
                        <div class="auto-refresh-toggle">
                            <span>Auto-refresh</span>
                            <div class="toggle-switch active" onclick="toggleAutoRefresh()" id="autoRefreshToggle">
                                <div class="toggle-slider"></div>
                            </div>
                        </div>
                        <button class="refresh-btn" onclick="loadCallLogs()" id="refreshBtn">🔄 Refresh Now</button>
                    </div>
                </div>
                <div id="callLogsContent">
                    <div class="loading">Loading call logs...</div>
                </div>
                <div class="last-updated" id="lastUpdated"></div>
            </div>
        </div>

        <script>
            let autoRefreshInterval = null;
            let autoRefreshEnabled = true;

            function parseTimeToSeconds(timeStr) {
                // Parse HH:MM:SS format
                const parts = timeStr.split(':');
                if (parts.length === 3) {
                    return parseInt(parts[0]) * 3600 + parseInt(parts[1]) * 60 + parseInt(parts[2]);
                }
                return 0;
            }

            function formatSecondsToTime(totalSeconds) {
                const hours = Math.floor(totalSeconds / 3600);
                const minutes = Math.floor((totalSeconds % 3600) / 60);
                const seconds = totalSeconds % 60;
                
                return `${hours.toString().padStart(2, '0')}:${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
            }

            function updateStats(data) {
                const totalCalls = data.reduce((sum, item) => sum + (item.call_count || 0), 0);
                const totalSeconds = data.reduce((sum, item) => sum + parseTimeToSeconds(item.total_time), 0);
                const avgSeconds = totalCalls > 0 ? Math.round(totalSeconds / totalCalls) : 0;
                const activeUsers = data.length;

                document.getElementById('totalCalls').textContent = totalCalls;
                document.getElementById('totalTime').textContent = formatSecondsToTime(totalSeconds);
                document.getElementById('activeUsers').textContent = activeUsers;
                document.getElementById('avgCallTime').textContent = formatSecondsToTime(avgSeconds);
            }

            function renderCallLogs(data) {
                const container = document.getElementById('callLogsContent');
                
                if (!data || data.length === 0) {
                    container.innerHTML = '<div class="no-data">No call logs found for today.</div>';
                    return;
                }

                const html = data.map(item => `
                    <div class="call-log-item">
                        <div class="user-info">
                            <div class="user-name">${item.name}</div>
                        </div>
                        <div class="call-stats">
                            <div class="stat-item">
                                <div class="stat-item-value">${item.call_count || 0}</div>
                                <div class="stat-item-label">Calls</div>
                            </div>
                            <div class="stat-item">
                                <div class="stat-item-value">${item.total_time}</div>
                                <div class="stat-item-label">Total Time</div>
                            </div>
                            <div class="stat-item">
                                <div class="stat-item-value">${item.last_call_ago}</div>
                                <div class="stat-item-label">Last Call</div>
                            </div>
                        </div>
                    </div>
                `).join('');

                container.innerHTML = html;
            }

            async function loadCallLogs() {
                const container = document.getElementById('callLogsContent');
                const refreshBtn = document.getElementById('refreshBtn');
                const lastUpdated = document.getElementById('lastUpdated');
                
                container.innerHTML = '<div class="loading">Loading call logs...</div>';
                refreshBtn.disabled = true;
                refreshBtn.textContent = '🔄 Loading...';

                try {
                    const response = await fetch('/api/call-logs');
                    
                    if (!response.ok) {
                        throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                    }
                    
                    const data = await response.json();
                    renderCallLogs(data);
                    updateStats(data);
                    
                    const now = new Date();
                    lastUpdated.textContent = `Last updated: ${now.toLocaleTimeString()}`;
                    
                } catch (error) {
                    console.error('Error loading call logs:', error);
                    container.innerHTML = `<div class="error">Failed to load call logs: ${error.message}</div>`;
                } finally {
                    refreshBtn.disabled = false;
                    refreshBtn.textContent = '🔄 Refresh Now';
                }
            }

            function toggleAutoRefresh() {
                const toggle = document.getElementById('autoRefreshToggle');
                autoRefreshEnabled = !autoRefreshEnabled;
                
                if (autoRefreshEnabled) {
                    toggle.classList.add('active');
                    startAutoRefresh();
                } else {
                    toggle.classList.remove('active');
                    stopAutoRefresh();
                }
            }

            function startAutoRefresh() {
                if (autoRefreshInterval) {
                    clearInterval(autoRefreshInterval);
                }
                
                autoRefreshInterval = setInterval(() => {
                    if (autoRefreshEnabled) {
                        loadCallLogs();
                    }
                }, 120000); // 2 minutes
            }

            function stopAutoRefresh() {
                if (autoRefreshInterval) {
                    clearInterval(autoRefreshInterval);
                    autoRefreshInterval = null;
                }
            }

            // Load data when page loads and start auto-refresh
            document.addEventListener('DOMContentLoaded', () => {
                loadCallLogs();
                startAutoRefresh();
            });

            // Clean up on page unload
            window.addEventListener('beforeunload', stopAutoRefresh);
        </script>
    </body>
    </html>
    """)

@app.route('/api/call-logs', methods=['GET'])
@limiter.limit("5 per minute")
def get_call_logs():
    """API endpoint to fetch call logs"""
    try:
        call_logs = fetch_call_logs()
        return jsonify(call_logs)
    except Exception as e:
        logger.error(f"API error: {str(e)}")
        return jsonify({'error': str(e)}), 500

@app.route('/health', methods=['GET'])
def health_check():
    """Health check endpoint"""
    return jsonify({'status': 'healthy', 'timestamp': datetime.utcnow().isoformat()}), 200

# Security headers middleware
@app.after_request
def after_request(response):
    response.headers['X-Content-Type-Options'] = 'nosniff'
    response.headers['X-Frame-Options'] = 'DENY'
    response.headers['X-XSS-Protection'] = '1; mode=block'
    response.headers['Referrer-Policy'] = 'strict-origin-when-cross-origin'
    if request.is_secure:
        response.headers['Strict-Transport-Security'] = 'max-age=31536000; includeSubDomains'
    return response

if __name__ == '__main__':
    # For development only - use proper WSGI server in production
    app.run(
        host='127.0.0.1',
        port=5000,
        debug=True
    )