<?php
// rc_client.php — JWT authentication + simple GET helper for RingCentral

$config = include __DIR__ . '/config.php';

function rc_client() {
    $token = rc_get_access_token();
    if (!$token) {
        throw new Exception("JWT authentication failed. Check debug.log for details.");
    }

    return new class($token) {
        private $accessToken;
        private $baseUrl;

        public function __construct($accessToken) {
            global $config;
            $this->accessToken = $accessToken;
            $this->baseUrl = rtrim($config['server'] ?? 'https://platform.ringcentral.com', '/');
        }

        public function get($path, $params = []) {
            $url = $this->baseUrl . $path;
            if ($params) {
                $url .= '?' . http_build_query($params);
            }

            $ch = curl_init($url);
            curl_setopt_array($ch, [
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_HTTPHEADER => [
                    'Authorization: Bearer ' . $this->accessToken,
                    'Accept: application/json'
                ],
                CURLOPT_TIMEOUT => 30,
            ]);
            $resp = curl_exec($ch);
            $err  = curl_error($ch);
            $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);

            if ($err) {
                file_put_contents(__DIR__ . '/debug.log', "[ERROR] cURL: $err\n", FILE_APPEND);
                throw new Exception("cURL error: $err");
            }

            $json = json_decode($resp, true);
            if ($code >= 400) {
                file_put_contents(__DIR__ . '/debug.log', "[ERROR] HTTP $code: $resp\n", FILE_APPEND);
                throw new Exception("HTTP $code: $resp");
            }
            return $json;
        }
    };
}

function rc_get_access_token() {
    global $config;

    $url = $config['server'] . '/restapi/oauth/token';
    $headers = [
        'Content-Type: application/x-www-form-urlencoded',
        'Authorization: Basic ' . base64_encode($config['clientId'] . ':' . $config['clientSecret'])
    ];

    $body = http_build_query([
        'grant_type' => 'urn:ietf:params:oauth:grant-type:jwt-bearer',
        'assertion'  => $config['jwt']
    ]);

    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_POSTFIELDS => $body,
        CURLOPT_TIMEOUT => 20
    ]);

    $resp = curl_exec($ch);
    $err  = curl_error($ch);
    curl_close($ch);

    if ($err) {
        file_put_contents(__DIR__ . '/debug.log', "Curl error: $err\n", FILE_APPEND);
        return null;
    }

    $json = json_decode($resp, true);
    if (isset($json['access_token'])) {
        file_put_contents(__DIR__ . '/debug.log', "[INFO] Token OK\n", FILE_APPEND);
        return $json['access_token'];
    } else {
        file_put_contents(__DIR__ . '/debug.log', "[ERROR] Token failed: $resp\n", FILE_APPEND);
        return null;
    }
}
?>
