<?php
/**
 * get_call_logs.php
 * ------------------
 * Returns today's call stats for all internal RingCentral users (agents).
 * Inbound calls are attributed to the agent receiving them.
 * Outbound calls are attributed to the agent making them.
 */

header('Content-Type: application/json');

// --- Safety settings: prevent HTML error output ---
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/php_errors.log');
error_reporting(E_ALL);

// --- Dependencies ---
require_once __DIR__ . '/rc_client.php';

// Optional helper file (safe include)
$utilPath = __DIR__ . '/util.php';
if (file_exists($utilPath)) {
    require_once $utilPath;
}

try {
    $config = include __DIR__ . '/config.php';
} catch (Throwable $e) {
    echo json_encode(['status' => 'error', 'message' => 'Failed to load config: ' . $e->getMessage()]);
    exit;
}

// --- Authenticate via JWT ---
$token = rc_get_access_token();
if (!$token) {
    echo json_encode(['status' => 'error', 'message' => 'Unable to authenticate with RingCentral']);
    exit;
}

// --- STEP 1: Get list of internal RingCentral agents ---
$usersUrl = $config['server'] . '/restapi/v1.0/account/~/extension';
$headers = [
    "Authorization: Bearer $token",
    "Accept: application/json"
];

$ch = curl_init($usersUrl);
curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_HTTPHEADER => $headers,
    CURLOPT_TIMEOUT => 20
]);
$resp = curl_exec($ch);
curl_close($ch);

if (!$resp) {
    echo json_encode(['status' => 'error', 'message' => 'Failed to retrieve RingCentral users']);
    exit;
}

$extData = json_decode($resp, true);
$internalUsers = [];
if (isset($extData['records'])) {
    foreach ($extData['records'] as $rec) {
        if (($rec['type'] ?? '') === 'User' && ($rec['status'] ?? '') === 'Enabled') {
            $internalUsers[$rec['id']] = $rec['name'];
        }
    }
}

if (empty($internalUsers)) {
    echo json_encode(['status' => 'error', 'message' => 'No internal RingCentral users found']);
    exit;
}

// --- STEP 2: Fetch today’s call logs ---
$startOfDay = date('c', strtotime('today midnight'));
$logsUrl = $config['server'] . '/restapi/v1.0/account/~/call-log?dateFrom=' . urlencode($startOfDay) . '&perPage=1000';

$ch = curl_init($logsUrl);
curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_HTTPHEADER => $headers,
    CURLOPT_TIMEOUT => 30
]);
$resp = curl_exec($ch);
curl_close($ch);

if (!$resp) {
    echo json_encode(['status' => 'error', 'message' => 'Failed to retrieve call logs']);
    exit;
}

$logData = json_decode($resp, true);
if (!isset($logData['records'])) {
    echo json_encode(['status' => 'error', 'message' => 'No call records found']);
    exit;
}

// --- STEP 3: Process logs and compute stats ---
$stats = [];

foreach ($logData['records'] as $call) {
    $direction = $call['direction'] ?? '';
    $duration  = (int)($call['duration'] ?? 0);
    $agentId = null;

    // Determine which agent this call belongs to
    if ($direction === 'Outbound' && isset($call['from']['extensionId'])) {
        // Outbound: use the calling agent’s extension
        $agentId = $call['from']['extensionId'];
    } elseif ($direction === 'Inbound') {
        // Inbound: try to find who answered internally
        if (isset($call['to']['extensionId'])) {
            $agentId = $call['to']['extensionId'];
        } elseif (!empty($call['legs']) && is_array($call['legs'])) {
            foreach ($call['legs'] as $leg) {
                if (
                    isset($leg['to']['extensionId']) &&
                    isset($leg['action']) && strtolower($leg['action']) === 'phone call' &&
                    isset($leg['result']) && strtolower($leg['result']) === 'answered'
                ) {
                    $agentId = $leg['to']['extensionId'];
                    break;
                }
            }
        }
    }

    // Skip calls not mapped to internal agents
    if (!$agentId || !isset($internalUsers[$agentId])) {
        continue;
    }

    $name = $internalUsers[$agentId];

    if (!isset($stats[$agentId])) {
        $stats[$agentId] = [
            'name' => $name,
            'inbound_count' => 0,
            'outbound_count' => 0,
            'inbound_time' => 0,
            'outbound_time' => 0
        ];
    }

    if ($direction === 'Inbound') {
        $stats[$agentId]['inbound_count']++;
        $stats[$agentId]['inbound_time'] += $duration;
    } elseif ($direction === 'Outbound') {
        $stats[$agentId]['outbound_count']++;
        $stats[$agentId]['outbound_time'] += $duration;
    }
}

// --- STEP 4: Format clean JSON output ---
$output = [];
foreach ($stats as $data) {
    $total = $data['inbound_time'] + $data['outbound_time'];
    $output[] = [
        'name' => $data['name'],
        'inbound_count' => $data['inbound_count'],
        'outbound_count' => $data['outbound_count'],
        'inbound_time' => gmdate("H:i:s", $data['inbound_time']),
        'outbound_time' => gmdate("H:i:s", $data['outbound_time']),
        'total_time' => gmdate("H:i:s", $total)
    ];
}

// Sort by total call time descending
usort($output, function ($a, $b) {
    $aTime = strtotime("1970-01-01 {$a['total_time']} UTC");
    $bTime = strtotime("1970-01-01 {$b['total_time']} UTC");
    return $bTime - $aTime;
});

// --- STEP 5: Return JSON ---
echo json_encode([
    'status' => 'success',
    'data' => $output,
    'count' => count($output)
], JSON_PRETTY_PRINT);

exit;
?>
