<?php
// Database configuration
$dbPath = __DIR__ . '/sponsor_content.db';

// Check if SQLite database exists
if (!file_exists($dbPath)) {
    die("Error: Database file not found at: $dbPath\n");
}

try {
    // Connect to SQLite database
    $db = new PDO('sqlite:' . $dbPath);
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    echo "Connected to database: " . realpath($dbPath) . "\n\n";
    
    // Check if sponsors table exists
    $tables = $db->query("SELECT name FROM sqlite_master WHERE type='table'")->fetchAll(PDO::FETCH_COLUMN);
    
    if (!in_array('sponsors', $tables)) {
        die("Error: 'sponsors' table not found in the database.\n");
    }
    
    // Get table structure
    echo "Sponsors table structure:\n";
    $result = $db->query("PRAGMA table_info('sponsors')");
    $columns = $result->fetchAll(PDO::FETCH_ASSOC);
    
    echo str_pad("Column", 20) . "Type    Null    Default\n";
    echo str_repeat("-", 50) . "\n";
    
    $hasBusinessName = false;
    foreach ($columns as $col) {
        echo str_pad($col['name'], 20) . 
             str_pad($col['type'], 8) . 
             ($col['notnull'] ? 'NO' : 'YES') . "     " . 
             ($col['dflt_value'] ?? 'NULL') . "\n";
             
        if ($col['name'] === 'business_name') {
            $hasBusinessName = true;
        }
    }
    
    // Check for business_name column
    if (!$hasBusinessName) {
        echo "\nWARNING: The 'business_name' column is missing from the sponsors table.\n";
        echo "Current columns: " . implode(', ', array_column($columns, 'name')) . "\n";
        
        // Show sample data
        $sample = $db->query("SELECT * FROM sponsors LIMIT 1")->fetch(PDO::FETCH_ASSOC);
        if ($sample) {
            echo "\nSample data from first sponsor:\n";
            print_r($sample);
        }
    }
    
} catch (PDOException $e) {
    die("Database error: " . $e->getMessage() . "\n");
}