<?php

namespace App;

use PDO;
use PDOException;

class Database {
    private $db;
    private $dbPath;
    
    public function __construct($dbPath = null) {
        if ($dbPath === null) {
            // Use the path from Config class
            $this->dbPath = __DIR__ . '/../' . Config::PATH_TO_SQLITE_FILE;
            
            // Ensure the directory exists
            $dir = dirname($this->dbPath);
            if (!file_exists($dir)) {
                mkdir($dir, 0777, true);
            }
        } else {
            $this->dbPath = $dbPath;
        }
        
        // Log the database path for debugging
        error_log("Database path: " . $this->dbPath);
        
        try {
            // Create the database file if it doesn't exist
            if (!file_exists($this->dbPath)) {
                touch($this->dbPath);
            }
            
            // Check if the database file is writable
            if (!is_writable($this->dbPath)) {
                throw new \Exception("Database file is not writable: " . $this->dbPath);
            }
            
            $this->db = new PDO('sqlite:' . $this->dbPath);
            $this->db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            
            // Check if the database is accessible
            $this->db->exec('PRAGMA foreign_keys = ON');
            
            // Initialize the database
            $this->initializeDatabase();
            
        } catch (\Exception $e) {
            error_log("Database connection error: " . $e->getMessage());
            throw $e;
        }
    }
    
    private function initializeDatabase() {
        // Create sponsors table if it doesn't exist
        $this->db->exec("CREATE TABLE IF NOT EXISTS sponsors (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            business_name TEXT,
            contract TEXT,
            address_1 TEXT,
            address_2 TEXT,
            street TEXT,
            city TEXT,
            county TEXT,
            postal_code TEXT,
            country TEXT,
            phone TEXT,
            mobile TEXT,
            email TEXT,
            url TEXT,
            social TEXT,
            banner TEXT,
            button TEXT,
            mp4 TEXT,
            sort_order INTEGER DEFAULT 0,
            is_active INTEGER DEFAULT 1,
            display_order INTEGER DEFAULT 0,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");
    }
    
    public function getAllSponsors() {
        $stmt = $this->db->query('SELECT id, business_name, contract, address_1, address_2, street, city, county, postal_code, country, phone, mobile, email, url, social, banner, button, mp4, sort_order, is_active, display_order, created_at, updated_at FROM sponsors ORDER BY display_order ASC, business_name ASC');
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    public function getSponsor($id) {
        $stmt = $this->db->prepare('SELECT * FROM sponsors WHERE id = :id');
        $stmt->execute(['id' => $id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }
    
    public function createSponsor($data) {
        try {
            $stmt = $this->db->prepare('INSERT INTO sponsors (
                business_name, contract, address_1, address_2, street, city, 
                county, postal_code, country, phone, mobile, email, url, 
                social, banner, button, mp4, sort_order, is_active, display_order
            ) VALUES (
                :business_name, :contract, :address_1, :address_2, :street, :city, 
                :county, :postal_code, :country, :phone, :mobile, :email, :url, 
                :social, :banner, :button, :mp4, :sort_order, :is_active, :display_order
            )');
            
            $result = $stmt->execute([
                'business_name' => $data['business_name'] ?? '',
                'contract' => $data['contract'] ?? '',
                'address_1' => $data['address_1'] ?? '',
                'address_2' => $data['address_2'] ?? '',
                'street' => $data['street'] ?? '',
                'city' => $data['city'] ?? '',
                'county' => $data['county'] ?? '',
                'postal_code' => $data['postal_code'] ?? '',
                'country' => $data['country'] ?? '',
                'phone' => $data['phone'] ?? '',
                'mobile' => $data['mobile'] ?? '',
                'email' => $data['email'] ?? '',
                'url' => $data['url'] ?? '',
                'social' => $data['social'] ?? '',
                'banner' => $data['banner'] ?? '',
                'button' => $data['button'] ?? '',
                'mp4' => $data['mp4'] ?? '',
                'sort_order' => $data['sort_order'] ?? 0,
                'is_active' => $data['is_active'] ?? 1,
                'display_order' => $data['display_order'] ?? 0
            ]);
            
            return $result ? $this->db->lastInsertId() : false;
            
        } catch (PDOException $e) {
            error_log('Error creating sponsor: ' . $e->getMessage());
            return false;
        }
    }
    
    public function updateSponsor($id, $data) {
        $sql = 'UPDATE sponsors SET 
                business_name = :business_name,
                contract = COALESCE(:contract, contract),
                address_1 = COALESCE(:address_1, address_1),
                address_2 = COALESCE(:address_2, address_2),
                street = COALESCE(:street, street),
                city = COALESCE(:city, city),
                county = COALESCE(:county, county),
                postal_code = COALESCE(:postal_code, postal_code),
                country = COALESCE(:country, country),
                phone = COALESCE(:phone, phone),
                mobile = COALESCE(:mobile, mobile),
                email = COALESCE(:email, email),
                url = COALESCE(:url, url),
                social = COALESCE(:social, social),
                banner = COALESCE(:banner, banner),
                button = COALESCE(:button, button),
                mp4 = COALESCE(:mp4, mp4),
                sort_order = :sort_order,
                is_active = :is_active,
                display_order = :display_order,
                updated_at = CURRENT_TIMESTAMP
                WHERE id = :id';
        
        $stmt = $this->db->prepare($sql);
        return $stmt->execute([
            ':id' => $id,
            ':business_name' => $data['business_name'] ?? '',
            ':contract' => $data['contract'] ?? '',
            ':address_1' => $data['address_1'] ?? '',
            ':address_2' => $data['address_2'] ?? '',
            ':street' => $data['street'] ?? '',
            ':city' => $data['city'] ?? '',
            ':county' => $data['county'] ?? '',
            ':postal_code' => $data['postal_code'] ?? '',
            ':country' => $data['country'] ?? '',
            ':phone' => $data['phone'] ?? '',
            ':mobile' => $data['mobile'] ?? '',
            ':email' => $data['email'] ?? '',
            ':url' => $data['url'] ?? '',
            ':social' => $data['social'] ?? '',
            ':banner' => $data['banner'] ?? '',
            ':button' => $data['button'] ?? '',
            ':mp4' => $data['mp4'] ?? '',
            ':sort_order' => $data['sort_order'] ?? 0,
            ':is_active' => $data['is_active'] ?? 1,
            ':display_order' => $data['display_order'] ?? 0
        ]);
    }
    
    public function deleteSponsor($id) {
        $stmt = $this->db->prepare('DELETE FROM sponsors WHERE id = ?');
        return $stmt->execute([$id]);
    }
    
    public function getActiveSponsors() {
        $stmt = $this->db->query('SELECT * FROM sponsors WHERE is_active = 1 ORDER BY display_order ASC, business_name ASC');
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
}
